'use client';

import { useState, useEffect } from 'react';

import { Team } from '../types';

import { NavBar, Footer } from '../components';
import TableContainer from '@mui/material/TableContainer';
import { Card, Typography } from '@mui/material';
import CardContent from '@mui/material/CardContent';

export default function HomepageDashboard() {
    const [teams, setTeams] = useState<Team[]>([]);

    const [inErrorState, setInErrorState] = useState(false);
    const [errorCode, setErrorCode] = useState('');

    async function fetchTeams() {
        try {
            const response = await fetch(`${process.env.NEXT_PUBLIC_BASE_API_URL}/teams`);
            const data = await response.json();
            if (data.length > 0) {
                setTeams(data);
            } else {
                setInErrorState(true);
                setErrorCode('Network connection error - data is missing for all Teams. Please confirm API is running and port 7275 is unblocked, then try again.');
            }
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        } catch (error: any) {
            console.error('Error fetching teams: ', error);
            setInErrorState(true);
            setErrorCode(error);
        }
    }

    useEffect(() => {
        // eslint-disable-next-line react-hooks/set-state-in-effect
        fetchTeams();
    }, [])

    function Teams() {
        return teams.map((team, index) => {
            return (
                <tr key={index} className="border-b border-gray-100 hover:bg-gray-50">
                    <td className="py-3 px-4">
                        <a href={`./teams/${team.teamid}`} className="text-[#0000ee]">
                            {team.name} <b>[{team.abbr}]</b>
                        </a>
                    </td>
                    <td className="py-3 px-4 font-medium">{team.city ? team.city : "-"}</td>
                    <td className="py-3 px-4 text-gray-600">{team.leagueid == 1 ? 'American' : 'National'} League</td>
                </tr>
            );
        });
    }

    return (
        <>
            <NavBar />
            <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 p-8" style={{ overflow: 'auto', marginTop: 50 }}>
                {inErrorState ?
                    <div className="flex content-center">
                        <Card sx={{ maxWidth: 345 }}>
                            <CardContent className="bg-red-100">
                                <Typography>
                                    <b>ERROR:</b> {errorCode}
                                </Typography>
                            </CardContent>
                        </Card>
                    </div>
                    :
                    <></>
                }
                <div className="max-w-6xl mx-auto">
                    <div className="bg-white rounded-lg shadow-lg p-6 mb-6">
                        <h1 className="text-4xl">All Teams:</h1>
                        <TableContainer>
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b-2 border-gray-200">
                                        <th className="text-left py-3 px-4 font-semibold text-gray-700">Name:</th>
                                        <th className="text-left py-3 px-4 font-semibold text-gray-700">City:</th>
                                        <th className="text-left py-3 px-4 font-semibold text-gray-700">League:</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {Teams()}
                                </tbody>
                            </table>
                        </TableContainer>
                    </div>
                </div>
            </div>
            <Footer />
        </>
    )
}
